<?php
/**
 * 文件：/api/404-guard.php  （修正版：避免 T_ENCAPSED 报错 + 静默容错）
 * 用途：作为 Nginx error_page 404 的“软 404 守卫”入口。
 * 触发方式：仅通过 Nginx 内部重写触发；直接访问可用 ?selftest=1 自检。
 */

@header('X-Soft-404-Guard: on');
@header('Cache-Control: no-store'); // 避免错误被缓存

// —— 自测模式：/api/404-guard.php?selftest=1 —— //
if (isset($_GET['selftest'])) {
    $info = [
        'ok'          => true,
        'cwd'         => getcwd(),
        '__DIR__'     => __DIR__,
        'root_guess'  => dirname(__DIR__),
        'request_uri' => isset($_GET['u']) ? (string)$_GET['u'] : '(none)',
        'php'         => PHP_VERSION,
    ];
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($info, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

// —— 解析原始请求路径 —— //
$reqUri  = isset($_GET['u']) ? (string)$_GET['u'] : (isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '/');
$reqPath = parse_url($reqUri, PHP_URL_PATH);
if (!$reqPath) $reqPath = '/';

// —— 根目录推断 —— //
$ROOT = dirname(__DIR__); // 如结构不同可直接改为固定路径

// —— 读取数据库配置（容错：找不到就降级仅 404） —— //
$dbConf = loadDbConfig($ROOT);
$pdo = null;

if (is_array($dbConf) && isset($dbConf['hostname'], $dbConf['database'], $dbConf['username'])) {
    // ★ 这里用“字符串拼接”，不要在双引号里写 $dbConf['xx']（会触发你遇到的解析错误）
    $host    = (string)$dbConf['hostname'];
    $port    = isset($dbConf['hostport']) ? (string)$dbConf['hostport'] : '3306';
    $dbname  = (string)$dbConf['database'];
    $charset = isset($dbConf['charset']) ? (string)$dbConf['charset'] : 'utf8mb4';

    $dsn = 'mysql:host=' . $host . ';port=' . $port . ';dbname=' . $dbname . ';charset=' . $charset;

    try {
        $pdo = new PDO(
            $dsn,
            (string)$dbConf['username'],
            isset($dbConf['password']) ? (string)$dbConf['password'] : '',
            [
                PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            ]
        );
    } catch (Throwable $e) {
        // 连不上库 → 降级
        $pdo = null;
    }
}

// —— 连接 Redis（可选） —— //
$redis = null;
try {
    if (class_exists('Redis')) {
        $redis = new Redis();
        // 如有密码/其他主机自行修改
        $redis->connect('127.0.0.1', 6379, 0.25);
        // $redis->auth('yourpass');
    }
} catch (Throwable $e) {
    $redis = null;
}

// ========== 1) 先查缓存：命中即 301 ==========
$cacheKey = 'redir:' . $reqPath;
if ($redis) {
    try {
        $hit = $redis->get($cacheKey);
        if ($hit) {
            jump($hit, 301, true);
        }
    } catch (Throwable $e) {}
}

// ========== 2) 查映射表 mac_redirect（确定性 301） ==========
if ($pdo) {
    try {
        $stmt = $pdo->prepare('SELECT new_path,http_code FROM mac_redirect WHERE old_path = ? LIMIT 1');
        $stmt->execute([$reqPath]);
        $row = $stmt->fetch();
        if ($row && !empty($row['new_path'])) {
            $target = (string)$row['new_path'];
            $code   = (int)(isset($row['http_code']) ? $row['http_code'] : 301);
            if ($code <= 0) $code = 301;
            if ($redis) { safe_redis_setex($redis, $cacheKey, 86400 * 7, $target); }
            jump($target, $code, true);
        }
    } catch (Throwable $e) {
        // 忽略，进入智能匹配
    }
}

// ========== 3) 智能匹配（仅对常见详情/播放路径） ==========
/* 关闭模糊跳转
$target = null;
if ($pdo) {
    $target = smartMatch($reqPath, $pdo);
    if ($target) {
        if ($redis) { safe_redis_setex($redis, $cacheKey, 86400 * 3, $target); }
        jump($target, 302, false); // 猜测 → 先 302
    }
}
*/

// ========== 4) 全部失败 → 软 404 回退 ==========
soft404();


// ===== 工具函数 ===== //

function safe_redis_setex($redis, $key, $ttl, $val) {
    try { $redis->setex($key, $ttl, $val); } catch (Throwable $e) {}
}

function jump($newPath, $code = 301, $permanent = true) {
    if ($permanent && (int)$code === 301) {
        header('Cache-Control: public, max-age=86400');
    } else {
        header('Cache-Control: no-store, max-age=0');
        header('X-Robots-Tag: noindex, nofollow');
    }
    if (strpos($newPath, 'http') !== 0) {
        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host   = isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'localhost';
        $newPath = $scheme . '://' . $host . $newPath;
    }
    header("Location: {$newPath}", true, (int)$code);
    exit;
}

function soft404() {
    http_response_code(404);
    header('Content-Type: text/html; charset=utf-8');
    header('X-Robots-Tag: noindex, nofollow');

    // 这个 URL 要能正确“渲染”你的 404 模板（返回 200 页面）
    // 建议做成 /label/404.html（苹果CMS会照常解析模板标签）
    $renderUrl = 'https://*.com/label/404.html';

    // 0.8 秒超时，避免卡顿；并附带一个头方便日后在 Nginx/应用侧排查
    $ctx = stream_context_create([
        'http' => [
            'method'  => 'GET',
            'timeout' => 1,
            'header'  => "X-From-Guard: 1\r\n",
            'ignore_errors' => true
        ]
    ]);

    $html = @file_get_contents($renderUrl, false, $ctx);

    if ($html !== false && strlen($html) > 256) {
        echo $html;   // 输出渲染后的完整页面，但 HTTP 状态仍是 404
        exit;
    }

    // 回退兜底：若上面失败，再尝试读你给的文件（纯 HTML,根据实际情况修改）
    $file = '/www/wwwroot/*com/template/mxpro/html/public/404.html';
    if (is_file($file) && is_readable($file)) {
        readfile($file);
    } else {
        echo '<!doctype html><meta charset="utf-8"><title>404</title><h1>页面不见了</h1><p>试试站内搜索或返回首页。</p>';
    }
    exit;
}


/**
 * 智能匹配：/voddetail/{id}.html 与 /vodplay/{id}-{sid}-{nid}.html
 * 成功返回相对路径，失败返回 null
 */
function smartMatch($path, $pdo) {
    // 详情页
    if (preg_match('#^/voddetail/(\d+)\.html$#i', $path, $m)) {
        $oldId = (int)$m[1];

        // ★ 强烈建议改造成“按同豆瓣ID找新片”的 SQL，这里仅做兜底示例
        $row = guessByNeighbor($oldId, $pdo);
        if ($row && !empty($row['vod_id'])) {
            return '/voddetail/' . (int)$row['vod_id'] . '.html';
        }
        return null;
    }

    // 播放页
    if (preg_match('#^/vodplay/(\d+)-(\d+)-(\d+)\.html$#i', $path, $m)) {
        $oldId = (int)$m[1];
        $sid   = (int)$m[2];
        $nid   = (int)$m[3];
        $row = guessByNeighbor($oldId, $pdo);
        if ($row && !empty($row['vod_id'])) {
            return '/vodplay/' . (int)$row['vod_id'] . '-' . $sid . '-' . $nid . '.html';
        }
        return null;
    }

    return null;
}

/**
 * 兜底猜测（示例）：挑最近更新时间的一条有效视频。
 * 实战中请替换成“同豆瓣ID”的精确跳转。
 */
function guessByNeighbor($oldId, $pdo) {
    try {
        $sql = "SELECT vod_id
                  FROM mac_vod
                 WHERE vod_status = 1
              ORDER BY vod_time DESC
                 LIMIT 1";
        $row = $pdo->query($sql)->fetch();
        return $row ?: null;
    } catch (Throwable $e) {
        return null;
    }
}

/**
 * 加载数据库配置（容错）：
 * - 优先：$ROOT/application/extra/database.php（MacCMS v10 默认）
 * - 兼容：$ROOT/application/database.php（部分 ThinkPHP 布局）
 * - 兼容：$ROOT/config/database.php（部分新布局）
 * 找不到 → 返回 null（降级仅 404）
 */
function loadDbConfig($ROOT) {
    $candidates = [
        $ROOT . '/application/extra/database.php',
        $ROOT . '/application/database.php',
        $ROOT . '/config/database.php',
    ];
    foreach ($candidates as $f) {
        if (is_file($f)) {
            $cfg = @include $f;
            if (is_array($cfg)) return $cfg;
        }
    }
    return null;
}
