<?php
namespace app\admin\controller;

use think\Db;
use think\Request;

class MergeVod extends Base
{
    /* ============ 页面：点击“筛选”才扫描 ============ */
    public function index(Request $request)
    {
        try {
            $this->ensureIgnoreTable();

            $kw     = trim((string)$request->param('kw', ''));
            $apply  = (int)$request->param('apply/d', 0);
            $rules  = $this->readRulesFromReq($request);
            $hasRules = $this->hasActiveRule($rules);
            $page   = max(1, (int)$request->param('page', 1));
            $limit  = max(10, (int)$request->param('limit', 20));
            $cats   = $this->readCatsFromReq($request);
            list($statusFilter, $statusRaw) = $this->readStatusFilter($request);
            $show   = $this->readShowFilter($request); // active / ignored / all

            $ignoredSet = $this->loadIgnoredKeys(); // ['key'=>1, ...]

            $groupsMap = [];
            if ($apply && $hasRules) {
                $groupsMap = $this->buildGroupMap($kw, $rules, $cats, $statusFilter);

                // 显示过滤
                $groupsMap = array_filter($groupsMap, function($g) use ($show, $ignoredSet){
                    $isIgnored = isset($ignoredSet[$g['key']]);
                    if ($show === 'ignored') return $isIgnored;
                    if ($show === 'active')  return !$isIgnored;
                    return true; // all
                });
            }

            $all   = array_values($groupsMap);
            $total = count($all);
            $offset   = ($page - 1) * $limit;
            $pageData = array_slice($all, $offset, $limit);

            // 计算建议目标（按规则挑）
            foreach ($pageData as &$g) {
                $g['target']  = $this->pickTargetVod($g, $rules);
                $g['ignored'] = isset($ignoredSet[$g['key']]) ? 1 : 0;
                // ★ 新增：建议目标在列表展示用“清洗后的名称”
                if (!empty($g['target'])) {
                    $g['target']['vod_name_display'] = (!empty($rules['name']) && isset($g['attrs']['name']))
                        ? $g['attrs']['name']
                        : $g['target']['vod_name'];
                }
            }

            // 传参
            $this->assign('kw', $kw);
            $this->assign('apply', $apply);
            $this->assign('rules', $rules);
            $this->assign('page', $page);
            $this->assign('limit', $limit);
            $this->assign('total', $total);
            $this->assign('groups', $pageData);
            $this->assign('cats_str', implode(',', $cats));
            $this->assign('status', $statusRaw);
            $this->assign('show', $show);
            $this->assign('rules_selected', $hasRules ? 1 : 0);

            // 模板兜底
            $viewSuffix = config('template.view_suffix') ?: 'html';
            $candidates = [];
            $candidates[] = rtrim(APP_PATH, '/\\') . '/admin/view/merge_vod/index.' . $viewSuffix;
            $viewPath = config('template.view_path');
            if ($viewPath) $candidates[] = rtrim($viewPath, '/\\') . '/merge_vod/index.' . $viewSuffix;
            if (defined('APPLICATION_PATH')) $candidates[] = rtrim(APPLICATION_PATH, '/\\') . '/admin/view/merge_vod/index.' . $viewSuffix;
            if (defined('ROOT_PATH')) $candidates[] = rtrim(ROOT_PATH, '/\\') . '/application/admin/view/merge_vod/index.' . $viewSuffix;

            foreach ($candidates as $tpl) {
                if (is_file($tpl)) return $this->fetch($tpl);
            }
            try { return $this->fetch('merge_vod/index'); }
            catch (\Throwable $e) {
                echo "模板文件未找到，已尝试：<br>" . implode('<br>', array_map('htmlspecialchars', $candidates)); exit;
            }
        } finally {
            try { \think\Db::close(); } catch (\Throwable $e) {}
        }
    }

    /* ============ 预览：支持分组内“候选条目多选” ============ */
    public function preview(Request $request)
    {
        try {
            $keys  = $request->post('groups/a', []);
            $rules = $this->readRulesFromReq($request);
            if (!$this->hasActiveRule($rules)) {
                return json(['code'=>1, 'msg'=>'请先在筛选区勾选入库重复规则']);
            }
            $cats  = $this->readCatsFromReq($request);
            list($statusFilter,) = $this->readStatusFilter($request);

            // 分组内选择映射：{ "key1":[id1,id2], ... }
            $selMap = [];
            $json = $request->post('sel_json', '');
            if ($json !== '') {
                $tmp = json_decode($json, true);
                if (is_array($tmp)) $selMap = $tmp;
            }

            if (empty($keys)) return json(['code'=>1, 'msg'=>'未选择任何分组']);
            $kw   = trim((string)$request->param('kw', ''));
            $groupsMap = $this->buildGroupMap($kw, $rules, $cats, $statusFilter);
            $report = [];

            foreach ($keys as $key) {
                if (!isset($groupsMap[$key])) continue;
                $g = $groupsMap[$key];

                $items = $g['items'];
                // 仅保留前端勾选的条目
                if (!empty($selMap[$key]) && is_array($selMap[$key])) {
                    $selIds = array_map('intval', $selMap[$key]);
                    $items = array_values(array_filter($items, function($r) use ($selIds){
                        return in_array((int)$r['vod_id'], $selIds, true);
                    }));
                }
                if (count($items) < 2) continue;

                $tmpG = $g; $tmpG['items'] = $items;
                $target  = $this->pickTargetVod($tmpG, $rules);
                if (!$target) continue;

                $sources = array_values(array_filter($items, function($r) use ($target) {
                    return (int)$r['vod_id'] !== (int)$target['vod_id'];
                }));

                $merged = $this->previewMergePlayData($target, $sources);

                $report[] = [
                    'key'      => $key,
                    'key_label'=> $g['key_label'],
                    'attrs'    => $g['attrs'],
                    'target'   => ['vod_id'=>$target['vod_id'], 'vod_name'=>$target['vod_name']],
                    'sources'  => array_map(function($r){ return ['vod_id'=>$r['vod_id'], 'vod_name'=>$r['vod_name']]; }, $sources),
                    'stats'    => $merged['stats'],
                    'sample'   => $merged['sample'],
                ];
            }

            return json(['code'=>0, 'data'=>$report]);
        } finally {
            try { \think\Db::close(); } catch (\Throwable $e) {}
        }
    }

    /* ============ 执行：支持“候选多选”，并沿用忽略功能 ============ */
    public function run(Request $request)
    {
        try {
            $keys    = $request->post('groups/a', []);
            $offline = (int)$request->post('offline_sources/d', 1) ? 1 : 0;
            $delete  = (int)$request->post('delete_sources/d', 0) ? 1 : 0;
            // 互斥：如前端意外传来同时为 1，优先“删除”，强制下线为 0
            if ($delete && $offline) { $offline = 0; }
            $rules   = $this->readRulesFromReq($request);
            if (!$this->hasActiveRule($rules)) {
                return json(['code'=>1, 'msg'=>'请先在筛选区勾选入库重复规则']);
            }
            $cats    = $this->readCatsFromReq($request);
            list($statusFilter,) = $this->readStatusFilter($request);

            $selMap = [];
            $json = $request->post('sel_json', '');
            if ($json !== '') {
                $tmp = json_decode($json, true);
                if (is_array($tmp)) $selMap = $tmp;
            }

            if (empty($keys)) return json(['code'=>1, 'msg'=>'未选择任何分组']);

            $kw   = trim((string)$request->param('kw', ''));
            $groupsMap = $this->buildGroupMap($kw, $rules, $cats, $statusFilter);

            $ok=0; $fail=0; $logs=[];

            foreach ($keys as $key) {
                if (!isset($groupsMap[$key])) { $fail++; $logs[] = "{$key} 未在重建分组中找到，跳过"; continue; }
                $g = $groupsMap[$key];

                // 限制到用户勾选的候选条目
                $items = $g['items'];
                if (!empty($selMap[$key]) && is_array($selMap[$key])) {
                    $selIds = array_map('intval', $selMap[$key]);
                    $items = array_values(array_filter($items, function($r) use ($selIds){
                        return in_array((int)$r['vod_id'], $selIds, true);
                    }));
                }

                if (count($items) < 2) { $fail++; $logs[] = $g['key_label']."（选择数不足2条），跳过"; continue; }

                $tmpG = $g; $tmpG['items'] = $items;
                $target = $this->pickTargetVod($tmpG, $rules);
                if (!$target) { $fail++; $logs[] = $g['key_label'].' 未找到目标'; continue; }

                $sources = array_values(array_filter($items, function($r) use ($target) {
                    return (int)$r['vod_id'] !== (int)$target['vod_id'];
                }));

                Db::startTrans();
                try {
                    // 锁目标
                    $t = Db::name('vod')->where('vod_id', $target['vod_id'])->lock(true)->find();
                    if (!$t) throw new \RuntimeException('目标不存在');

                    $basePlay = $this->explodePlay($t['vod_play_from'], $t['vod_play_url'], $t['vod_play_note']);

                    // 批量取来源播放三字段，避免 N 次查库
                    $ids = array_column($sources, 'vod_id');
                    $srcRows = [];
                    if (!empty($ids)) {
                        $list = Db::name('vod')
                            ->where('vod_id', 'in', $ids)
                            ->field('vod_id, vod_play_from, vod_play_url, vod_play_note')
                            ->select();
                        foreach ($list as $row) {
                            $srcRows[(int)$row['vod_id']] = $row;
                        }
                    }

                    $stats = ['from_added'=>0, 'ep_added'=>0, 'dup_skipped'=>0];
                    foreach ($sources as $s) {
                        $sv = $srcRows[(int)$s['vod_id']] ?? null;
                        if (!$sv) continue;
                        $inc = $this->explodePlay($sv['vod_play_from'], $sv['vod_play_url'], $sv['vod_play_note']);
                        $this->mergePlayArrays($basePlay, $inc, $stats);
                    }
                    list($pf, $pu, $pn) = $this->implodePlay($basePlay);

                    $upd = [
                        'vod_play_from' => $pf,
                        'vod_play_url'  => $pu,
                        'vod_play_note' => $pn,
                    ];
                    if (!empty($rules['name'])) $upd['vod_name'] = $g['attrs']['name'] ?? $t['vod_name'];
                    if (!empty($rules['year'])) $upd['vod_year'] = $g['attrs']['year'] ?? $t['vod_year'];

                    Db::name('vod')->where('vod_id', $target['vod_id'])->update($upd);

                    if (!empty($ids)) {
                        if ($delete) {
                            Db::name('vod')->where('vod_id', 'in', $ids)->delete();
                        } elseif ($offline) {
                            Db::name('vod')->where('vod_id', 'in', $ids)->update(['vod_status'=>0]);
                        }
                    }

                    Db::commit();
                    $ok++;
                    $logs[] = $g['key_label']." ✔ 目标ID {$target['vod_id']}，来源 ".implode(',', $ids)
                           ."；新增播放器{$stats['from_added']}，新增分集{$stats['ep_added']}，跳过重复{$stats['dup_skipped']}"
                           . ($delete ? '；已删除来源条目' : ($offline ? '；已下线来源条目' : ''));
                } catch (\Throwable $e) {
                    Db::rollback();
                    $fail++;
                    $logs[] = $g['key_label']." ✖ 失败：".$e->getMessage();
                }
            }

            return json(['code'=>0, 'msg'=>"完成：成功{$ok}，失败{$fail}", 'logs'=>$logs]);
        } finally {
            try { \think\Db::close(); } catch (\Throwable $e) {}
        }
    }

    /* ======= 忽略/取消忽略 接口（POST） ======= */
    public function ignore(Request $request)
    {
        try {
            $this->ensureIgnoreTable();

            $key  = (string)$request->post('key', '');
            $flag = (int)$request->post('flag/d', 1); // 1 忽略、0 取消
            $note = trim((string)$request->post('note', ''));

            if ($key === '') return json(['code'=>1, 'msg'=>'缺少 key']);

            try {
                if ($flag) {
                    // upsert
                    $hit = Db::name('merge_vod_ignore')->where('grp_key', $key)->find();
                    if ($hit) {
                        Db::name('merge_vod_ignore')->where('id', $hit['id'])->update(['note'=>$note]);
                    } else {
                        Db::name('merge_vod_ignore')->insert(['grp_key'=>$key, 'note'=>$note]);
                    }
                } else {
                    Db::name('merge_vod_ignore')->where('grp_key', $key)->delete();
                }
                return json(['code'=>0, 'msg'=>$flag ? '已忽略' : '已取消忽略']);
            } catch (\Throwable $e) {
                return json(['code'=>1, 'msg'=>$e->getMessage()]);
            }
        } finally {
            try { \think\Db::close(); } catch (\Throwable $e) {}
        }
    }

    /* ================== 规则/筛选读取 ================== */
private function readRulesFromReq(Request $req)
{
    $arr = $req->param('rules/a', []);
    $rules = [
        'name'     => in_array('name', $arr) ? 1 : 0,
        'category' => in_array('category', $arr) ? 1 : 0,
        'year'     => in_array('year', $arr) ? 1 : 0,
        'area'     => in_array('area', $arr) ? 1 : 0,
        'lang'     => in_array('lang', $arr) ? 1 : 0,
        'actor'    => in_array('actor', $arr) ? 1 : 0,
        'director' => in_array('director', $arr) ? 1 : 0,
        'douban'   => in_array('douban', $arr) ? 1 : 0,
    ];
    // 不再兜底任何规则；若全空，后续 buildGroupMap 会返回空结果，提示用户选择规则
    return $rules;
}
    //新增 hasActiveRule 辅助方法并在构建分组及前端模板中使用，提示用户未勾选规则的情况
    private function hasActiveRule(array $rules): bool
    {
        foreach ($rules as $flag) {
            if (!empty($flag)) {
                return true;
            }
        }
        return false;
    }

    private function readCatsFromReq(Request $req): array
    {
        $raw = trim((string)$req->param('cats', ''));
        if ($raw === '') return [];
        $arr = preg_split('/[,\s]+/', $raw, -1, PREG_SPLIT_NO_EMPTY);
        $ids = [];
        foreach ($arr as $x) { $v = (int)$x; if ($v>0) $ids[$v]=1; }
        return array_keys($ids);
    }
    
    private function readStatusFilter(Request $req): array
    {
        $raw = trim((string)$req->param('status', '1'));
        if ($raw === '') {
            return [null, ''];
        }
        if ($raw === '1' || $raw === '0') {
            return [(int)$raw, $raw];
        }
        return [1, '1'];
    }
    
    private function readShowFilter(Request $req): string
    {
        $v = strtolower(trim((string)$req->param('show', 'active')));
        if (!in_array($v, ['active','ignored','all'], true)) $v = 'active';
        return $v;
    }

    /* ================== 忽略存取 ================== */
    private function ensureIgnoreTable()
    {
        // 简单检测：若不存在则创建（首次运行）
        try {
            Db::name('merge_vod_ignore')->count();
        } catch (\Throwable $e) {
            // 尝试创建
            $sql = "CREATE TABLE IF NOT EXISTS `".config('database.prefix')."merge_vod_ignore` (
                `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
                `grp_key` VARCHAR(255) NOT NULL,
                `note` VARCHAR(255) NOT NULL DEFAULT '',
                `ctime` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `uniq_grp_key` (`grp_key`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='合并工具-忽略分组'";
            Db::execute($sql);
        }
    }
    private function loadIgnoredKeys(): array
    {
        $ret = [];
        try {
            $rows = Db::name('merge_vod_ignore')->field('grp_key')->select();
            foreach ($rows as $r) { $ret[$r['grp_key']] = 1; }
        } catch (\Throwable $e) { /* ignore */ }
        return $ret;
    }

    /* ============= 分组/目标/播放合并等：保持你现有实现 ============= */

private function buildGroupMap($kw = '', $rules = [], $catIds = [], $status = null)
{
    // 若未勾选任何规则，直接返回空（由上层提示“请选择入库重复规则”）
    if (!$this->hasActiveRule($rules)) {
        return [];
    }

    // 1) 候选集合：按 kw（片名关键字）与可选分类过滤
    $rowsQ = Db::name('vod')
        ->field('vod_id, vod_name, vod_year, type_id_1, vod_area, vod_lang, vod_actor, vod_director, vod_douban_id, vod_status, vod_play_from, vod_play_url, vod_play_note');

    if ($kw !== '') {
        $rowsQ->where('vod_name', 'like', "%{$kw}%");
    }
    if (!empty($catIds)) {
        $rowsQ->where('type_id_1', 'in', $catIds);
    }
    if ($status !== null) {
        $rowsQ->where('vod_status', '=', (int)$status);
    }

    // 如果既没有 kw 也没有分类范围，为了避免全表扫描，可以按需加一个“最近入库”限制（可选）
    // $rowsQ->order('vod_id desc')->limit(50000);

    $rows = $rowsQ->order('vod_id desc')->select();

    // 2) 分组：仅根据“勾选的规则字段”取 attrs 作为 key
    $groups = [];
    foreach ($rows as $r) {
        $attrs = [];

        if (!empty($rules['name'])) {
            // 清洗：去掉片名结尾的“(年份)/(年份)/[年份]/空格+年份”；如果标题无年份就用原名
            list($clean, $yFromTitle) = $this->extractYearAndClean($r['vod_name']);
            if ($clean === '') continue;
            $attrs['name'] = $clean;
        }
        if (!empty($rules['category'])) {
            $cat = (int)$r['type_id_1']; if (!$cat) continue;
            $attrs['category'] = $cat;
        }
        if (!empty($rules['year'])) {
            // 注意：这里采用“数据库里的年份”，即使标题不带年份也可参与分组
            $yearDb = (int)$r['vod_year']; if (!$yearDb) continue;
            $attrs['year'] = $yearDb;
        }
        if (!empty($rules['area'])) {
            $area = $this->normText($r['vod_area']); if ($area==='') continue;
            $attrs['area'] = $area;
        }
        if (!empty($rules['lang'])) {
            $lang = $this->normText($r['vod_lang']); if ($lang==='') continue;
            $attrs['lang'] = $lang;
        }
        if (!empty($rules['actor'])) {
            $actor = $this->normList($r['vod_actor']); if ($actor==='') continue;
            $attrs['actor'] = $actor;
        }
        if (!empty($rules['director'])) {
            $director = $this->normList($r['vod_director']); if ($director==='') continue;
            $attrs['director'] = $director;
        }
        if (!empty($rules['douban'])) {
            $douban = trim((string)$r['vod_douban_id']); if ($douban==='') continue;
            $attrs['douban'] = $douban;
        }

        // 没有任何可用属性，则跳过
        if (empty($attrs)) continue;

        // 组键 & 显示标签
        $keyParts = []; $labelParts = [];
        foreach (['name'=>'名称','category'=>'分类','year'=>'年份','area'=>'地区','lang'=>'语言','actor'=>'演员','director'=>'导演','douban'=>'豆瓣ID'] as $k=>$label){
            if (!empty($rules[$k]) && isset($attrs[$k])) {
                $keyParts[]   = $k.':'.$attrs[$k];
                $labelParts[] = $label.'：'.htmlspecialchars(is_string($attrs[$k])?$attrs[$k]:(string)$attrs[$k], ENT_QUOTES);
            }
        }
        if (empty($keyParts)) continue;

        $key = implode('||', $keyParts);
        if (!isset($groups[$key])) {
            $groups[$key] = [
                'key'       => $key,
                'key_label' => implode(' | ', $labelParts),
                'attrs'     => $attrs,
                'items'     => [],
            ];
        }
        $groups[$key]['items'][] = $r;
    }

    // 3) 仅保留 2 条及以上
    $groups = array_filter($groups, function($g){ return count($g['items']) >= 2; });

    // 4) 排序：数量降序 → 组键
    uasort($groups, function($a,$b){
        $c = count($b['items']) - count($a['items']);
        if ($c !== 0) return $c;
        return strcmp($a['key'], $b['key']);
    });

    return $groups;
}


    /**
     * 仅使用分组内 items 挑选目标，避免额外查库：
     * 1) 勾了“名称”→ 优先：标题不含年份 且 完全等于 clean name 的条目；若勾“年份”，再优先 vod_year 一致（通常已一致）
     * 2) 兜底：标题最短，其次 ID 最小
     */
    private function pickTargetVod(array $group, array $rules)
    {
        $attrs = $group['attrs'];
        $items = $group['items'];

        if (!empty($rules['name']) && isset($attrs['name'])) {
            $clean = mb_strtolower(trim($attrs['name']));
            $cands = [];
            foreach ($items as $r) {
                list($_c, $_y) = $this->extractYearAndClean($r['vod_name']);
                $noYearInTitle = !$_y;
                $nameEq = (mb_strtolower(trim($r['vod_name'])) === $clean);
                if ($noYearInTitle && $nameEq) {
                    $score = 0;
                    if (!empty($rules['year']) && isset($attrs['year']) && (int)$r['vod_year'] === (int)$attrs['year']) {
                        $score += 10;
                    }
                    $cands[] = [$score, mb_strlen($r['vod_name']), (int)$r['vod_id'], $r];
                }
            }
            if (!empty($cands)) {
                usort($cands, function($a,$b){
                    if ($a[0] !== $b[0]) return $b[0]-$a[0];   // score DESC
                    if ($a[1] !== $b[1]) return $a[1]-$b[1];   // name length ASC
                    return $a[2]-$b[2];                        // id ASC
                });
                return $cands[0][3];
            }
        }

        // 兜底：标题最短，其次 ID 最小
        usort($items, function($a,$b){
            $la = mb_strlen($a['vod_name']); $lb = mb_strlen($b['vod_name']);
            if ($la !== $lb) return $la - $lb;
            return $a['vod_id'] - $b['vod_id'];
        });
        return $items[0] ?? null;
    }

    /* ===== 其余：播放列解析/合并/清洗（与现有一致） ===== */

    private function explodePlay($from, $url, $note)
    {
        $fromArr = (string)$from === '' ? [] : explode('$$$', (string)$from);
        $urlArr  = (string)$url  === '' ? [] : explode('$$$', (string)$url);

        $ret = ['froms'=>[], 'order'=>[]];
        foreach ($fromArr as $idx => $fm) {
            $fm = trim($fm);
            if ($fm === '') continue;
            $ret['order'][] = $fm;
            $ret['froms'][$fm] = [];

            $uStr = isset($urlArr[$idx]) ? (string)$urlArr[$idx] : '';
            if ($uStr === '') continue;

            $eps = explode('#', $uStr);
            foreach ($eps as $ep) {
                $ep = trim($ep);
                if ($ep === '') continue;
                $pos = mb_strpos($ep, '$');
                if ($pos === false) {
                    $ret['froms'][$fm][] = ['n'=>'', 'u'=>trim($ep)];
                } else {
                    $n = trim(mb_substr($ep, 0, $pos));
                    $u = trim(mb_substr($ep, $pos+1));
                    $ret['froms'][$fm][] = ['n'=>$n, 'u'=>$u];
                }
            }
        }
        return $ret;
    }
    
    
     /* ===== 同名播放器已存在时，不再合并该播放器下的任何分集 ===== */
    private function mergePlayArrays(&$base, $inc, &$stats)
    {
        if (!isset($base['froms'])) $base['froms'] = [];
        if (!isset($base['order'])) $base['order'] = [];
    
        foreach ($inc['order'] as $fm) {
            if (!isset($inc['froms'][$fm])) continue;
            $incList = $inc['froms'][$fm];
    
            // 目标中不存在该播放器：整体新增播放器及其所有分集
            if (!isset($base['froms'][$fm])) {
                $base['froms'][$fm] = [];
                $base['order'][] = $fm;
                $stats['from_added']++;
    
                foreach ($incList as $it) {
                    $base['froms'][$fm][] = ['n'=>$it['n'], 'u'=>$it['u']];
                    $stats['ep_added']++;
                }
            } else {
                // 目标已有同名播放器 → 不合并该播放器下的分集
                // 如需统计“跳过分集数”，可放开下一行：
                // $stats['dup_skipped'] += count($incList);
                continue;
            }
        }
    }

    
    private function epKey($n, $u)
    {
        $n = $this->normalizeEpName($n);
        $u = trim((string)$u);
        return md5($n . '||' . $u);
    }
    private function normalizeEpName($n)
    {
        $n = preg_replace('/\\s+/u', ' ', trim((string)$n));
        $n = preg_replace('/　+/u', ' ', $n);
        return $n;
    }
    private function implodePlay($arr)
    {
        $froms = $arr['froms'] ?? [];
        $order = $arr['order'] ?? [];

        $pf=[]; $pu=[]; $pn=[];
        foreach ($order as $fm) {
            $pf[] = $fm;
            $eps = $froms[$fm] ?? [];
            $epStrs = [];
            foreach ($eps as $it) {
                $n = trim($it['n']); $u = trim($it['u']);
                $epStrs[] = ($n === '' ? $u : ($n.'$'.$u));
            }
            $pu[] = implode('#', $epStrs);
            $pn[] = ''; // 保持对齐
        }
        return [implode('$$$', $pf), implode('$$$', $pu), implode('$$$', $pn)];
    }

    private function previewMergePlayData($target, $sources)
    {
        $base  = $this->explodePlay($target['vod_play_from'], $target['vod_play_url'], $target['vod_play_note']);
        $stats = ['from_added'=>0, 'ep_added'=>0, 'dup_skipped'=>0];

        foreach ($sources as $s) {
            $inc = $this->explodePlay($s['vod_play_from'], $s['vod_play_url'], $s['vod_play_note']);
            $this->mergePlayArrays($base, $inc, $stats);
        }

        $sample = [];
        $i = 0;
        foreach ($base['order'] as $fm) {
            if (!isset($base['froms'][$fm])) continue;
            $i++; if ($i > 3) break;
            $eps = $base['froms'][$fm];
            $sample[] = [
                'from'  => $fm,
                'count' => count($eps),
                'first' => $eps[0] ?? null,
            ];
        }

        return ['stats'=>$stats, 'sample'=>$sample];
    }

    /* 清洗片名 */
    private function extractYearAndClean($name)
    {
        $orig = $name = trim((string)$name);
        if (preg_match('/(（|\\(|\\[)\\s*(19\\d{2}|20\\d{2})\\s*年?\\s*(）|\\)|\\])\\s*$/u', $name, $m)) {
            $year  = (int)$m[2];
            $clean = preg_replace('/(（|\\(|\\[)\\s*(19\\d{2}|20\\d{2})\\s*年?\\s*(）|\\)|\\])\\s*$/u', '', $name);
            return [$this->rtrimPunct($clean), $year];
        }
        if (preg_match('/\\s(19\\d{2}|20\\d{2})\\s*年?\\s*$/u', $name, $m)) {
            $year  = (int)$m[1];
            $clean = preg_replace('/\\s(19\\d{2}|20\\d{2})\\s*年?\\s*$/u', '', $name);
            return [$this->rtrimPunct($clean), $year];
        }
        if (preg_match('/(19\\d{2}|20\\d{2})\\s*$/u', $name, $m)) {
            $year  = (int)$m[1];
            $clean = preg_replace('/(19\\d{2}|20\\d{2})\\s*$/u', '', $name);
            return [$this->rtrimPunct($clean), $year];
        }
        return [$orig, null];
    }
    private function rtrimPunct($s)
    {
        $s = preg_replace('/[\\s\\-_:|·、，,。.!！?？~～]+$/u', '', (string)$s);
        $s = preg_replace('/[（\\(\\[【]+$/u', '', $s);
        return trim($s);
    }
    
    /**
 * 规范化普通文本（用于 地区/语言 等）
 * - 去两端空白与分隔符
 * - 合并多空格
 * - 忽略“未知/内详”等占位
 * - 统一为小写做分组键
 */
private function normText($s): string
{
    $s = trim((string)$s);
    if ($s === '') return '';
    // 去常见无效占位
    $low = mb_strtolower($s, 'UTF-8');
    $invalid = ['未知','内详','無','不详','不詳','n/a','na','unknown','null'];
    if (in_array($low, $invalid, true)) return '';

    // 合并空白、去首尾常见分隔符
    $s = preg_replace('/\s+/u', ' ', $s);          // 多空白 -> 单空格
    $s = trim($s, " \t\n\r\0\x0B,，、;；|/·-");     // 去首尾分隔符
    if ($s === '') return '';

    return mb_strtolower($s, 'UTF-8');             // 统一小写，便于分组对比
}

/**
 * 规范化人物列表（用于 演员/导演 等）
 * - 支持 , ， ； ; | / 、 作为分隔符（不按空格拆，避免把英文名拆开）
 * - 去掉“主演：/演员：/导演：”等前缀
 * - 去重（忽略大小写）、排序，拼成稳定的键
 * - 返回形式：用 "||" 连接，作为分组键；若为空返回 ''
 */
private function normList($s): string
{
    $s = trim((string)$s);
    if ($s === '') return '';

    // 去常见前缀
    $s = preg_replace('/^(主演|演员|導演|导演)\s*[:：]\s*/u', '', $s);

    // 统一分隔符：, ， ； ; | / 、 -> 逗号
    $s = preg_replace('/\s*[,，;；|\/、]\s*/u', ',', $s);
    $s = trim($s, ','); // 去首尾逗号

    if ($s === '') return '';

    $parts = array_map('trim', explode(',', $s));

    $norm = [];
    foreach ($parts as $p) {
        if ($p === '') continue;
        // 忽略无效占位
        $low = mb_strtolower($p, 'UTF-8');
        if (in_array($low, ['未知','内详','無','不详','不詳','n/a','na','unknown','null'], true)) {
            continue;
        }
        // 去重（大小写不敏感）
        $key = $low;
        $norm[$key] = $p; // 保留原样（用于显示时更好看），键用小写去重
    }

    if (empty($norm)) return '';

    // 排序，保证次序无关
    ksort($norm, SORT_NATURAL);

    // 作为分组键：用 "||" 连接（稳定、避免与人名字符冲突）
    return implode('||', array_values($norm));
}

    
}
